package com.cmeeting.service.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.IdUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.cmeeting.ad.entity.RobotSecurityUser;
import com.cmeeting.ad.entity.SysUserSync;
import com.cmeeting.ad.util.SecurityUtil;
import com.cmeeting.constant.RecordTemplateConstant;
import com.cmeeting.dto.UserDTO;
import com.cmeeting.mapper.primary.CommonMapper;
import com.cmeeting.mapper.primary.MeetingRecordTemplateMapper;
import com.cmeeting.mapper.primary.RecordTemplatePermissionMapper;
import com.cmeeting.mapper.secondary.SysUserSysMapper;
import com.cmeeting.pojo.MeetingRecordTemplate;
import com.cmeeting.pojo.RecordTemplatePermission;
import com.cmeeting.service.MeetingRecordTemplateService;
import com.cmeeting.service.SysUserSyncService;
import com.cmeeting.util.MinioUtils;
import com.cmeeting.vo.RecordTemplateVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.InputStream;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
public class MeetingRecordTemplateServiceImpl extends ServiceImpl<MeetingRecordTemplateMapper, MeetingRecordTemplate>  implements MeetingRecordTemplateService {

    @Autowired
    private MinioUtils minioUtils;
    @Resource
    private RecordTemplatePermissionMapper permissionMapper;
    @Resource
    private SysUserSysMapper sysUserSysMapper;
    @Resource
    private SysUserSyncService sysUserSyncService;
    @Value(value = "${permission.tenantId}")
    private String permissionTenantId;

    @Override
    public IPage<MeetingRecordTemplate> getPage(RecordTemplateVO vo) {
        RobotSecurityUser user = SecurityUtil.getUser();
        LambdaQueryWrapper<MeetingRecordTemplate> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(MeetingRecordTemplate::getIsDel,Boolean.FALSE);
        queryWrapper.eq(StringUtils.isNotEmpty(vo.getType()),MeetingRecordTemplate::getType,vo.getType());
        queryWrapper.like(StringUtils.isNotEmpty(vo.getName()),MeetingRecordTemplate::getName,vo.getName());
        queryWrapper.like(StringUtils.isNotEmpty(vo.getMeetingType()),MeetingRecordTemplate::getMeetingType,vo.getMeetingType());
        queryWrapper.like(StringUtils.isNotEmpty(vo.getCreateUser()),MeetingRecordTemplate::getCreateUserName,vo.getCreateUser());
        queryWrapper.between(vo.getStartTime() != null && vo.getEndTime() != null,MeetingRecordTemplate::getUpdateTime,vo.getStartTime(),vo.getEndTime());
        queryWrapper.orderByDesc(MeetingRecordTemplate::getId);
        IPage<MeetingRecordTemplate> resultPage = baseMapper.selectPage(new Page<>(vo.getCurrent(), vo.getSize()), queryWrapper);
        List<MeetingRecordTemplate> records = resultPage.getRecords();
        if(!CollectionUtils.isEmpty(records)){
            List<SysUserSync> simpleUserList = sysUserSysMapper.getSimpleUserList(user.getTenantId());
            Map<String, String> simpleUserMap = CollectionUtils.isEmpty(simpleUserList)
                    ? new HashMap<>() : simpleUserList.stream().collect(Collectors.toMap(SysUserSync::getUserId, SysUserSync::getName));
            for (MeetingRecordTemplate record : records) {
                String createUser = record.getCreateUser();
                String createUserName = simpleUserMap.containsKey(createUser) ? simpleUserMap.get(createUser) : "1".equals(createUser) ? "集团" : createUser;
                record.setCreateUser(createUserName);
            }
        }
        resultPage.setRecords(records);
        return resultPage;
    }

    @Override
    public IPage<MeetingRecordTemplate> getPersonalPage(RecordTemplateVO vo) {
        RobotSecurityUser user = SecurityUtil.getUser();
        Integer current = vo.getCurrent();
        Integer size = vo.getSize();
        //查出所有系统模板
        LambdaQueryWrapper<MeetingRecordTemplate> systemQueryWrapper = new LambdaQueryWrapper<>();
        systemQueryWrapper.eq(MeetingRecordTemplate::getIsDel,Boolean.FALSE);
        systemQueryWrapper.eq(MeetingRecordTemplate::getType,RecordTemplateConstant.TEMPLATE_TYPE_SYSTEM);
        systemQueryWrapper.like(vo.getName() != null,MeetingRecordTemplate::getName,vo.getName());
        systemQueryWrapper.like(vo.getMeetingType() != null,MeetingRecordTemplate::getMeetingType,vo.getMeetingType());
        systemQueryWrapper.like(StringUtils.isNotEmpty(vo.getCreateUser()),MeetingRecordTemplate::getCreateUserName,vo.getCreateUser());
        systemQueryWrapper.between(vo.getStartTime() != null && vo.getEndTime() != null,MeetingRecordTemplate::getUpdateTime,vo.getStartTime(),vo.getEndTime());
        systemQueryWrapper.orderByDesc(MeetingRecordTemplate::getId);
        List<MeetingRecordTemplate> systemTemplateList = baseMapper.selectList(systemQueryWrapper);
        //用户可见的系统模板权限
        List<MeetingRecordTemplate> authorizedSystemTemplateList = new ArrayList<>();
        Map<Integer, List<String>> authorizedUserMap = selectAuthorizedUsers().stream().collect(Collectors.toMap(item -> item.getRecordTemplateId(), item -> item.getUserIdList()));
        for (MeetingRecordTemplate systemTemplate : systemTemplateList) {
            if(authorizedUserMap.containsKey(systemTemplate.getId())){
                List<String> authorizedUserIds = authorizedUserMap.get(systemTemplate.getId());
                if(authorizedUserIds.contains(user.getId()))
                    authorizedSystemTemplateList.add(systemTemplate);
            }
        }

        //查出用户个人的自定义模板
        LambdaQueryWrapper<MeetingRecordTemplate> customQueryWrapper = new LambdaQueryWrapper<>();
        customQueryWrapper.eq(MeetingRecordTemplate::getIsDel,Boolean.FALSE);
        customQueryWrapper.eq(MeetingRecordTemplate::getType,RecordTemplateConstant.TEMPLATE_TYPE_CUSTOM);
        customQueryWrapper.eq(MeetingRecordTemplate::getCreateUser, user.getId());
        customQueryWrapper.like(vo.getName() != null,MeetingRecordTemplate::getName,vo.getName());
        customQueryWrapper.like(vo.getMeetingType() != null,MeetingRecordTemplate::getMeetingType,vo.getMeetingType());
        customQueryWrapper.like(StringUtils.isNotEmpty(vo.getCreateUser()),MeetingRecordTemplate::getCreateUserName,vo.getCreateUser());
        customQueryWrapper.between(vo.getStartTime() != null && vo.getEndTime() != null,MeetingRecordTemplate::getUpdateTime,vo.getStartTime(),vo.getEndTime());
        customQueryWrapper.orderByDesc(MeetingRecordTemplate::getId);
        List<MeetingRecordTemplate> customTemplateList = baseMapper.selectList(customQueryWrapper);

        List<MeetingRecordTemplate> resultList = new ArrayList<>();
        resultList.addAll(authorizedSystemTemplateList);
        resultList.addAll(customTemplateList);
        List<MeetingRecordTemplate> records = getUsersByOffset(resultList, current, size);
        if(!CollectionUtils.isEmpty(records)){
            List<SysUserSync> simpleUserList = sysUserSysMapper.getSimpleUserList(user.getTenantId());
            Map<String, String> simpleUserMap = CollectionUtils.isEmpty(simpleUserList)
                    ? new HashMap<>() : simpleUserList.stream().collect(Collectors.toMap(SysUserSync::getUserId, SysUserSync::getName));
            for (MeetingRecordTemplate record : records) {
                String createUser = record.getCreateUser();
                String createUserName = simpleUserMap.containsKey(createUser) ? simpleUserMap.get(createUser) : "1".equals(createUser) ? "集团" : createUser;
                record.setCreateUser(createUserName);
            }
        }
        IPage<MeetingRecordTemplate> resultPage = new Page<>(current, size);
        resultPage.setTotal(resultList.size());
        resultPage.setRecords(records);
        resultPage.setPages((resultList.size() + size - 1) / size);
        return resultPage;
    }

    /**
     * 上传附件模板到minio
     * @param file
     * @param id
     * @return
     */
    @Override
    public String templateUpload(MultipartFile file, Integer id) {
        MeetingRecordTemplate detail = getById(id);
        String separator = "/";
        String point = ".";
        // 获取文件名
        String originalFilename = file.getOriginalFilename();
        // 获取文件后缀
        String originalSuffix = FileUtil.getSuffix(originalFilename);
        // 生成文件名
        String nowDate = DateUtil.format(new Date(), "yyyy-MM-dd");
        String uuid = IdUtil.fastSimpleUUID();
        String originalFilePath = separator + nowDate + separator + uuid + point + originalSuffix;
        // 获取文件流
        try(InputStream is = file.getInputStream()){
            minioUtils.upload(originalFilePath,is);
        }catch (Exception e){
            throw new RuntimeException(e.getMessage());
        }
        detail.setTemplate(originalFilePath);
        updateById(detail);
        return originalFilePath;
    }

    /**
     * 获取模板授权的人员
     * @return
     */
    @Override
    public List<UserDTO.TemplateAuthorizedUserDTO> selectAuthorizedUsers() {
        List<UserDTO.TemplateAuthorizedUserDTO> result = new ArrayList<>();
        //查出所有系统模板(暂定)
        List<MeetingRecordTemplate> meetingRecordTemplates = baseMapper.selectList(new LambdaQueryWrapper<MeetingRecordTemplate>().eq(MeetingRecordTemplate::getType, RecordTemplateConstant.TEMPLATE_TYPE_SYSTEM));
        //查出所有系统模板授权信息
        List<RecordTemplatePermission> permissions = permissionMapper.selectList(new LambdaQueryWrapper<RecordTemplatePermission>().eq(RecordTemplatePermission::getTemplateType, RecordTemplateConstant.TEMPLATE_TYPE_SYSTEM));
        //所有授权模板的人员
        for (MeetingRecordTemplate template : meetingRecordTemplates) {
            UserDTO.TemplateAuthorizedUserDTO resultItem = new UserDTO.TemplateAuthorizedUserDTO();
            resultItem.setRecordTemplate(template);
            resultItem.setRecordTemplateId(template.getId());
            //该系统模板授权部门
            List<RecordTemplatePermission> deptPermissions = permissions.stream().filter(item -> "1".equals(item.getRelType()) && template.getId().equals(item.getTemplateId())).collect(Collectors.toList());
            //该系统模板授权人员
            List<RecordTemplatePermission> userPermissions = permissions.stream().filter(item -> "2".equals(item.getRelType()) && template.getId().equals(item.getTemplateId())).collect(Collectors.toList());

            List<String> authorizedUsers = new ArrayList<>();
            if(!CollectionUtils.isEmpty(userPermissions)){
                for (RecordTemplatePermission userPermission : userPermissions) {
                    authorizedUsers.add(userPermission.getRelId());
                }
            }
            if(!CollectionUtils.isEmpty(deptPermissions)){
                List<String> deptPath = new ArrayList<>();
                for (RecordTemplatePermission deptPermission : deptPermissions) {
                    String deptId = deptPermission.getRelId();
                    getDeptPath(deptPath,deptId,permissionTenantId);
                }
                List<String> usersByDept = sysUserSysMapper.getUsersByDept(deptPath,permissionTenantId);
                if(!CollectionUtils.isEmpty(usersByDept))
                    authorizedUsers.addAll(usersByDept);
            }
            resultItem.setUserIdList(authorizedUsers);
            result.add(resultItem);
        }
        return result;
    }

    @Override
    public List<MeetingRecordTemplate> getSelectList(RecordTemplateVO vo) {
        RobotSecurityUser user = SecurityUtil.getUser();
        //查出所有系统模板
        LambdaQueryWrapper<MeetingRecordTemplate> systemQueryWrapper = new LambdaQueryWrapper<>();
        systemQueryWrapper.eq(MeetingRecordTemplate::getIsDel,Boolean.FALSE);
        systemQueryWrapper.eq(MeetingRecordTemplate::getType,RecordTemplateConstant.TEMPLATE_TYPE_SYSTEM);
        systemQueryWrapper.like(vo.getName() != null,MeetingRecordTemplate::getName,vo.getName());
        systemQueryWrapper.like(vo.getMeetingType() != null,MeetingRecordTemplate::getMeetingType,vo.getMeetingType());
        systemQueryWrapper.orderByDesc(MeetingRecordTemplate::getId);
        List<MeetingRecordTemplate> systemTemplateList = baseMapper.selectList(systemQueryWrapper);
        if(CollectionUtils.isEmpty(systemTemplateList)) return new ArrayList<>();
        //用户可见的系统模板权限
        List<MeetingRecordTemplate> records = new ArrayList<>();

        //查出所有系统模板授权人员信息
        List<RecordTemplatePermission> userPermissions = permissionMapper.selectList(new LambdaQueryWrapper<RecordTemplatePermission>()
                .eq(RecordTemplatePermission::getTemplateType, RecordTemplateConstant.TEMPLATE_TYPE_SYSTEM)
                .eq(RecordTemplatePermission::getRelId,user.getId())
                .eq(RecordTemplatePermission::getRelType,RecordTemplateConstant.REL_TYPE_USER)
        );
        //当前登录人员被直接授权的模板
        List<Integer> templateIdsAccepted = CollectionUtils.isEmpty(userPermissions)
                ? new ArrayList<>() : userPermissions.stream().map(RecordTemplatePermission::getTemplateId).collect(Collectors.toList());

        //查出所有系统模板授权部门信息
        List<RecordTemplatePermission> deptPermissions = permissionMapper.selectList(new LambdaQueryWrapper<RecordTemplatePermission>()
                        .eq(RecordTemplatePermission::getTemplateType, RecordTemplateConstant.TEMPLATE_TYPE_SYSTEM)
                        .eq(RecordTemplatePermission::getRelId,user.getId())
                        .eq(RecordTemplatePermission::getRelType,RecordTemplateConstant.REL_TYPE_DEPT));
        //查出当前登录人的部门向上路径
        String deptPathByUser = sysUserSyncService.getDeptPathByUserId(user.getId());
        if(StringUtils.isNotEmpty(deptPathByUser) && !CollectionUtils.isEmpty(deptPermissions)){
            List<Integer> templateIdsDeptAccepted = deptPermissions.stream().filter(item -> deptPathByUser.contains(item.getRelId()))
                    .map(RecordTemplatePermission::getTemplateId).collect(Collectors.toList());
            if(!CollectionUtils.isEmpty(templateIdsDeptAccepted)) templateIdsAccepted.addAll(templateIdsDeptAccepted);
        }
        for (MeetingRecordTemplate template : systemTemplateList) {
            if(templateIdsAccepted.contains(template.getId())){
                records.add(template);
            }
        }
        return records;
    }

    /**
     * 获取部门的路径
     * @param deptPath
     * @param deptId
     */
    private void getDeptPath(List<String> deptPath, String deptId, String tenantId) {
        if(!deptPath.contains(deptId)) deptPath.add(deptId);
        List<String> subDeptIds = sysUserSysMapper.getSubDeptId(deptId,tenantId);
        if(CollectionUtils.isEmpty(subDeptIds)) return;

        for (String subDeptId : subDeptIds) {
            //部门id去重
            if(!deptPath.contains(subDeptId)){
                deptPath.add(subDeptId);
                getDeptPath(deptPath,subDeptId,tenantId);
            }
        }

    }

    /**
     * 手动分页，偏移取每一页数据
     * @param list
     * @param current
     * @param size
     * @return
     */
    public List<MeetingRecordTemplate> getUsersByOffset(List<MeetingRecordTemplate> list, int current, int size) {
        int offset = (current - 1) * size;
        int endIndex = Math.min(offset + size, list.size());

        // 防止偏移量越界
        if (offset >= list.size()) {
            return Collections.emptyList();
        }

        return list.subList(offset, endIndex);
    }
}