package com.cmeeting.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.cmeeting.annotation.OperLog;
import com.cmeeting.constant.RecordTemplateConstant;
import com.cmeeting.dto.RecordTemplateDto;
import com.cmeeting.pojo.MeetingRecordTemplate;
import com.cmeeting.service.MeetingRecordTemplateService;
import com.cmeeting.service.RecordTemplatePermissionService;
import com.cmeeting.util.R;
import com.cmeeting.util.page.PageUtil;
import com.cmeeting.vo.EditRecordTemplateVo;
import com.cmeeting.vo.RecordTemplateVO;
import com.deepoove.poi.XWPFTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


/**
 * 纪要模板控制层
 */
@RestController
@RequestMapping("/recordTemplate")
public class RecordTemplateController {
    @Autowired
    private MeetingRecordTemplateService recordTemplateService;
    @Resource
    private RecordTemplatePermissionService recordTemplatePermissionService;

    /**
     * 创建模板
     */
    @OperLog(location = "模板页面", operation = "创建模板")
    @PostMapping("/create")
    public R create(@RequestBody @Validated(EditRecordTemplateVo.AddGroup.class) EditRecordTemplateVo recordTemplate) {

        Long id = recordTemplateService.save(recordTemplate);
        return R.ok(id);
    }

    /**
     * 更新模板
     */
    @OperLog(location = "模板页面", operation = "更新模板")
    @PostMapping("/update")
    public R update(@RequestBody @Validated(EditRecordTemplateVo.UpdateGroup.class) EditRecordTemplateVo recordTemplate) {

        recordTemplateService.updateById(recordTemplate);
        return R.ok();
    }

    /**
     * 删除模板
     */
    @OperLog(location = "模板页面", operation = "删除模板")
    @PostMapping("/delete")
    public R delete(@RequestBody @Validated(EditRecordTemplateVo.DeleteGroup.class) EditRecordTemplateVo recordTemplate) {
        recordTemplateService.delete(recordTemplate.getId());
        return R.ok();
    }

    /**
     * 启用模板
     */
    @OperLog(location = "模板页面", operation = "启用模板")
    @PostMapping("/enable")
    public R enable(@RequestBody @Validated(EditRecordTemplateVo.EnableGroup.class) EditRecordTemplateVo recordTemplate) {
        return R.ok(recordTemplateService.enable(recordTemplate.getId()));
    }

    /**
     * 获取会议模板下拉框，有权限控制
     * @return
     */
    @GetMapping("/select")
    public R select() {
        List<RecordTemplateDto> recordTemplateDtoList =recordTemplateService.getTemplateSelect();
        return R.ok(recordTemplateDtoList);
    }

    /**
     * 纪要模板列表，通过type区分为系统模板还是自定义模板
     * @param vo
     * @return
     */
    @PostMapping("/list")
    public R list(@RequestBody RecordTemplateVO vo) {
        Integer current = vo.getCurrent();
        Integer size = vo.getSize();
        PageUtil.startPage(size, current, "cmt_meeting_record_template.id desc");

        List<RecordTemplateDto> list = recordTemplateService.getRecordTemplateDtoList(vo);
        return PageUtil.getDataTable(list);
    }

    @GetMapping("/detail")
    public R detail(@RequestParam("id") Long id) {

        return R.ok(recordTemplateService.getById(id));
    }

    /**
     * 用户下拉选择的模板列表（权限内）
     * @param vo
     * @return
     */
    @PostMapping("/selectList")
    public R selectList(@RequestBody RecordTemplateVO vo) {
        List<MeetingRecordTemplate> records = recordTemplateService.getSelectList(vo);
        return R.ok(records);
    }

    /**
     * 普通用户纪要模板，查出有授权的系统模板和个人创建的自定义模板
     * @param vo
     * @return
     */
    @PostMapping("/personalList")
    public R personalList(@RequestBody RecordTemplateVO vo) {
        IPage<MeetingRecordTemplate> page = recordTemplateService.getPersonalPage(vo);
        return R.ok(page);
    }

    @PostMapping("/detail")
    public R detail(@RequestBody RecordTemplateVO vo) {
        MeetingRecordTemplate detail = recordTemplateService.getById(vo.getId());
        return R.ok(detail);
    }

    @PostMapping("/templateUpload")
    public R templateUpload(@RequestParam("file") MultipartFile file,@RequestParam("id") Integer id) {
        String template = recordTemplateService.templateUpload(file, id);
        return R.ok(template);
    }

    /**
     * 模板授权
     * @param vo
     * @return
     */
    @PostMapping("/auth")
    public R auth(@RequestBody RecordTemplateVO.TemplatePermissionVO vo) {
        return R.ok(recordTemplatePermissionService.auth(vo));
    }

    /**
     * 模板已授权列表
     * @param vo
     * @return
     */
    @PostMapping("/authList")
    public R authList(@RequestBody RecordTemplateVO.TemplateAuthVO vo) {
        return R.ok(recordTemplatePermissionService.authList(vo));
    }

    /**
     * 获取模板授权的人员
     * @return
     */
    @PostMapping("/selectAuthorizedUsers")
    public R selectAuthorizedUsers() {
        return R.ok(recordTemplateService.selectAuthorizedUsers());
    }

    /**
     * 模板测试效果
     * @param file 用户自主上传的转录文件
     * @param meetingInstId 历史会议主键id
     * @param content 模板提示词
     * @return
     */
    @PostMapping("/testGenerate")
    public R testGenerate(@RequestParam(value = "file",required = false) MultipartFile file,
                          @RequestParam(value = "meetingInstId",required = false) Integer meetingInstId,
                          @RequestParam("content") String content) {
        return R.ok(recordTemplatePermissionService.testGenerate(file, meetingInstId, content));
    }

    /**
     * 模板测试效果导出纪要
     * @param recordJson 纪要内容json
     * @param meetingInstId 历史会议主键id
     * @param id 模板id
     * @return
     */
    @PostMapping("/testMinutesExport")
    public void testMinutesExport(@RequestParam(value = "recordJson") String recordJson,
                                  @RequestParam(value = "meetingInstId",required = false) Integer meetingInstId,
                                  @RequestParam("id") Integer id,
                                  HttpServletResponse response) {
        recordTemplatePermissionService.testMinutesExport(recordJson, meetingInstId, id, response);
    }
}
