package com.cmeeting.service.impl;

import cn.chatbot.openai.completion.chat.ChatCompletionRequest;
import cn.chatbot.openai.completion.chat.ChatMessage;
import cn.chatbot.openai.completion.chat.ChatMessageRole;
import cn.chatbot.openai.completion.chat.Message;
import cn.chatbot.openai.service.LLMService;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.IoUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.poi.excel.ExcelUtil;
import cn.hutool.poi.excel.ExcelWriter;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.cmeeting.ad.entity.RobotSecurityUser;
import com.cmeeting.ad.util.SecurityUtil;
import com.cmeeting.constant.MeetingState;
import com.cmeeting.exception.RobotBaseException;
import com.cmeeting.log.service.ProcessLogService;
import com.cmeeting.mapper.primary.MeetingInfoMapper;
import com.cmeeting.mapper.primary.MeetingRecordTemplateMapper;
import com.cmeeting.mapper.primary.UserIdMapper;
import com.cmeeting.pojo.MeetingInfo;
import com.cmeeting.pojo.MeetingRecordTemplate;
import com.cmeeting.pojo.UserId;
import com.cmeeting.service.*;
import com.cmeeting.util.MinioUtils;
import com.cmeeting.vo.MeetingInfoVO;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.tencentcloudapi.wemeet.Client;
import com.tencentcloudapi.wemeet.core.authenticator.AuthenticatorBuilder;
import com.tencentcloudapi.wemeet.core.authenticator.JWTAuthenticator;
import com.tencentcloudapi.wemeet.service.meetings.api.MeetingsApi;
import com.tencentcloudapi.wemeet.service.meetings.model.V1MeetingsMeetingIdParticipantsGet200Response;
import com.tencentcloudapi.wemeet.service.meetings.model.V1MeetingsMeetingIdParticipantsGet200ResponseParticipantsInner;
import com.tencentcloudapi.wemeet.service.records.api.RecordsApi;
import com.tencentcloudapi.wemeet.service.records.model.V1AddressesRecordFileIdGet200Response;
import com.tencentcloudapi.wemeet.service.records.model.V1AddressesRecordFileIdGet200ResponseAiMeetingTranscriptsInner;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.math.BigInteger;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
public class MeetingInfoServiceImpl extends ServiceImpl<MeetingInfoMapper, MeetingInfo>  implements MeetingInfoService {

    @Resource
    private MeetingInfoMapper mapper;
    @Resource
    private MinioUtils minioUtils;
    @Value(value = "${tencent.appId}")
    private String tencentAppId;
    @Value(value = "${tencent.sdkId}")
    private String tencentSdkId;
    @Value(value = "${tencent.secretId}")
    private String tencentSecretId;
    @Value(value = "${tencent.secretKey}")
    private String tencentSecretKey;
    @Value(value = "${tencent.admin.userId}")
    private String tencentAdminUserId;
    @Value(value = "${llm.api-addr}")
    private String llmApiAddr;
    @Value(value = "${llm.api-model}")
    private String llmApiModel;
    @Value(value = "${llm.api-token}")
    private String llmApiToken;
    @Value(value = "${llm.api-max-tokens}")
    private Integer llmApiMaxTokens;
    @Value(value = "${tencent.base-save-path}")
    private String baseSavePath;
    @Resource
    private ProcessLogService processLogService;
    @Resource
    private MeetingRecordTemplateMapper meetingRecordTemplateMapper;
    @Resource
    private MeetingInfoMapper meetingInfoMapper;
    @Resource
    private UserIdMapper userIdMapper;
    @Resource
    private SysUserSyncService sysUserSyncService;

    @Override
    public IPage<MeetingInfo> getPage(MeetingInfoVO vo) {
        RobotSecurityUser user = SecurityUtil.getUser();
        String userId = user.getId();
        //根据员工号获取腾会uid
        String tid = userIdMapper.getTidByWid(userId);
        if(StringUtils.isEmpty(tid)){
            log.error("获取历史会议列表失败：根据员工号"+userId+"获取腾会uid失败");
            return new Page<>(0,0);
        }
        LambdaQueryWrapper<MeetingInfo> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(MeetingInfo::getHostUid,tid);
        queryWrapper.like(StringUtils.isNotEmpty(vo.getSubject()),MeetingInfo::getSubject,vo.getSubject());
        queryWrapper.between(vo.getStartTime() != null,MeetingInfo::getStartTime,vo.getStartTime(),vo.getEndTime());
        IPage<MeetingInfo> resultPage = mapper.selectPage(new Page<>(vo.getCurrent(), vo.getSize()), queryWrapper);
        return resultPage;
    }

    @Override
    public boolean updateRecordXml(MeetingInfoVO vo) {
        //前端的表单json转xml，xml格式的纪要内容便于生成会议纪要文件
        String recordJson = vo.getRecordJson();
        String recordXml = convertJSONToXml(recordJson);
        String key = vo.getMeetingId() + "-recordXmlPath-" + IdUtil.fastSimpleUUID() + ".xml";
        minioUtils.upload(key,recordXml.getBytes(StandardCharsets.UTF_8));
        vo.setRecordXml(key);

        LambdaUpdateWrapper<MeetingInfo> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(MeetingInfo::getId,vo.getId());
        updateWrapper.set(MeetingInfo::getUpdateTime,LocalDateTime.now());
        updateWrapper.set(MeetingInfo::getRecordXml,key);
        return update(null,updateWrapper);
    }

    @Override
    public boolean regenerateXml(MeetingInfoVO vo) {
        Client client = new Client.Builder()
                .withAppId(tencentAppId).withSdkId(tencentSdkId)
                .withSecret(tencentSecretId,tencentSecretKey)
                .build();
        MeetingInfo meetingInfo = mapper.selectById(vo.getId());
        //已保存的会议信息
        String meetingId = meetingInfo.getMeetingId();
        String subMeetingId = meetingInfo.getSubMeetingId();
        String recordFileIdArray = meetingInfo.getRecordFileId();
        try{
            if(StringUtils.isEmpty(recordFileIdArray)){
                return false;
            }
            String meetingDate = meetingInfo.getStartTime().toLocalDate().format(DateTimeFormatter.ISO_LOCAL_DATE);

            // 获取参会成员明细
            MeetingsApi.ApiV1MeetingsMeetingIdParticipantsGetRequest participantsRequest =
                    new MeetingsApi.ApiV1MeetingsMeetingIdParticipantsGetRequest
                            .Builder(meetingId).subMeetingId(subMeetingId).operatorId(tencentAdminUserId).operatorIdType("1").build();
            AuthenticatorBuilder<JWTAuthenticator> participantsAuthenticatorBuilder =
                    new JWTAuthenticator.Builder()
                            .nonce(BigInteger.valueOf(Math.abs((new SecureRandom()).nextInt())))
                            .timestamp(String.valueOf(System.currentTimeMillis() / 1000L));
            MeetingsApi.ApiV1MeetingsMeetingIdParticipantsGetResponse participantsResponse =
                    client.meetings().v1MeetingsMeetingIdParticipantsGet(participantsRequest, participantsAuthenticatorBuilder);
            V1MeetingsMeetingIdParticipantsGet200Response participantsData = participantsResponse.getData();
            List<V1MeetingsMeetingIdParticipantsGet200ResponseParticipantsInner> participants = participantsData.getParticipants();
            String participantNames = participants.stream().map(item -> new String(Base64.getDecoder().decode(item.getUserName()))).distinct().collect(Collectors.joining("、"));
            meetingInfo.setParticipantUsers(participantNames);

            //每场会议可能会分段录制，查出每个文件的转录记录后拼接
            StringBuffer recordTextBuffer = new StringBuffer();
            List<String> recordFileIdList = Arrays.asList(meetingInfo.getRecordFileId().split(","));
            for (String recordFileId : recordFileIdList) {
                //查询录制转写详情
                RecordsApi.ApiV1AddressesRecordFileIdGetRequest addressRequest =
                        new RecordsApi.ApiV1AddressesRecordFileIdGetRequest.Builder(recordFileId)
                                .operatorId(tencentAdminUserId)
                                .operatorIdType("1")
                                .build();
                RecordsApi.ApiV1AddressesRecordFileIdGetResponse addressResponse =
                        client.records().v1AddressesRecordFileIdGet(addressRequest,
                                new JWTAuthenticator.Builder().nonce(BigInteger.valueOf(Math.abs((new SecureRandom()).nextInt())))
                                        .timestamp(String.valueOf(System.currentTimeMillis() / 1000L)));
                // 处理响应
                if (addressResponse != null && addressResponse.getData() != null) {
                    log.info("Successfully got address for record file {}", recordFileId);
                    V1AddressesRecordFileIdGet200Response addressData = addressResponse.getData();

                    // 获取AI会议转录文件
                    List<V1AddressesRecordFileIdGet200ResponseAiMeetingTranscriptsInner> transcripts =
                            addressData.getAiMeetingTranscripts();

                    if (transcripts != null && !transcripts.isEmpty()) {
                        log.info("Found {} AI meeting transcripts for record file {}",
                                transcripts.size(), recordFileId);

                        // 处理每个转录文件
                        for (V1AddressesRecordFileIdGet200ResponseAiMeetingTranscriptsInner transcript : transcripts) {
                            String fileType = transcript.getFileType();
                            String downloadUrl = transcript.getDownloadAddress();

                            if ("txt".equalsIgnoreCase(fileType)) {

                                // 1. 下载文件
                                byte[] fileData = downloadFile(downloadUrl);

                                // 2. 将二进制文件转换为文本
                                String recordTextContent = new String(fileData);
                                if(StringUtils.isNotEmpty(recordTextContent.replaceAll("\\n","").trim())){
                                    recordTextBuffer.append("\n\n");
                                    recordTextBuffer.append(recordTextContent);
                                }
                            }
                        }
                    } else {
                        log.info("No AI meeting transcripts found for record file {}", recordFileId);
                    }
                } else {
                    log.warn("Empty response for record file: {}", recordFileId);
                }
            }
            if(StringUtils.isEmpty(recordTextBuffer.toString().replaceAll("\\n","").trim())){
                log.info("获取的转录文本为空，跳过纪要生成，meetingId:{}，fileRecordId:{}",meetingId,recordFileIdList.toString());
                processLogService.log(meetingId,subMeetingId,"获取的转录文本为空，跳过纪要生成");
                throw new RuntimeException("获取的转录文本为空，跳过纪要生成");
            }
            //获取系统模板
            MeetingRecordTemplate meetingRecordTemplate = meetingRecordTemplateMapper.selectById(2);
            String processedResult = processWithClaude(recordTextBuffer.toString(),meetingDate,meetingRecordTemplate.getPrompt(),meetingId,subMeetingId);
            saveResult(processedResult, recordTextBuffer.toString().getBytes(StandardCharsets.UTF_8), meetingInfo,  meetingRecordTemplate);
            meetingInfoMapper.update(null,
                    new LambdaUpdateWrapper<MeetingInfo>()
                            .eq(MeetingInfo::getMeetingRecordId, meetingInfo.getMeetingRecordId())
                            .eq(subMeetingId != null,MeetingInfo::getSubMeetingId,subMeetingId)
                            .set(MeetingInfo::getRecordXml,meetingInfo.getRecordXml()));
            return true;
        }catch (Exception e){
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            e.printStackTrace(pw);
            processLogService.log(meetingId,subMeetingId,sw.toString());
            return false;
        }
    }

    /**
     * 保存会议纪要相关的文件
     * @param content 大模型返回的不规则xml
     * @param recordData 转录文本
     * @param meetingInfo 会议对象
     * @param meetingRecordTemplate 模板信息
     */
    private void saveResult(String content, byte[] recordData, MeetingInfo meetingInfo,MeetingRecordTemplate meetingRecordTemplate) {
        String meetingId = meetingInfo.getMeetingId();
        String subMeetingId = meetingInfo.getSubMeetingId();
        //转录文件临时存储路径
        String recordContentPath = meetingId + "-recordContent-" + IdUtil.fastSimpleUUID() + ".txt";
        //生成的xml临时存储路径
        String recordXmlPath = meetingId + "-recordXmlPath-" + IdUtil.fastSimpleUUID() + ".xml";
        try {
            //去除内容中除了xml内容以外其他的信息，格式化xml
            String xml = extractXmlFromMarkdown(content, meetingId, subMeetingId);
            minioUtils.upload(recordXmlPath,xml.getBytes(StandardCharsets.UTF_8));
            meetingInfo.setRecordXml(recordXmlPath);
        } catch (Exception e) {
            log.error("填充会议纪要失败: {}", e.getMessage(), e);
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            e.printStackTrace(pw);
            processLogService.log(meetingId,subMeetingId,"填充会议纪要失败"+sw.toString());
            throw new RuntimeException("填充会议纪要失败");
        }
    }

    /**
     * 大模型生成纪要xml
     * @param textContent 转录文件
     * @param prompt 提示词
     * @return
     */
    private String processWithClaude(String textContent, String meetingDate, String prompt,String meetingId, String subMeetingId) {
        //将文件传送给大模型处理
        String token = "Bearer " + llmApiToken;
        String apiAddr = llmApiAddr;
        String model = llmApiModel;
        int maxTokens = llmApiMaxTokens;
        List<Message> messages = new ArrayList<>();

        ChatMessage chatMessage = new ChatMessage(ChatMessageRole.USER.value(), prompt.replaceAll("\\{transcript\\}",textContent).replaceAll("\\{meetingDate\\}",meetingDate));
        messages.add(chatMessage);
//        chatMessage = new ChatMessage(ChatMessageRole.ASSISTANT.value(), "好的请提供会议记录");
//        messages.add(chatMessage);
//        chatMessage = new ChatMessage(ChatMessageRole.USER.value(), textContent);

        messages.add(chatMessage);

        // 调用Claude API处理文件
        String ret = call_llm(apiAddr, model, token, messages, maxTokens,meetingId,subMeetingId);
        return ret;
    }

    private String call_llm(String apiAddr, String model, String token, List<Message> messages, int maxTokens,String meetingId, String subMeetingId) {
        LLMService service = new LLMService(token, apiAddr);
        StringBuilder stringBuilder = new StringBuilder();
        try {
            ChatCompletionRequest chatCompletionRequest = ChatCompletionRequest.builder()
                    .model(model)
                    .messages(messages)
                    .stream(true)
                    .n(1)
                    .maxTokens(maxTokens)
                    .logitBias(new HashMap<>())
                    .build();

            service.streamChatCompletion(chatCompletionRequest).doOnError(Throwable::printStackTrace).blockingForEach(chunk -> {
                chunk.getChoices().stream().map(choice -> choice.getMessage().getContent())
                        .filter(Objects::nonNull).findFirst().ifPresent(o -> {
                    try {
                        stringBuilder.append(new String(o.getBytes(Charset.defaultCharset())));
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                });
            });
        } catch (Exception e) {
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            e.printStackTrace(pw);
            processLogService.log(meetingId,subMeetingId,"【大模型处理异常】:"+sw.toString());
            throw new RuntimeException(e);
        }
        service.shutdownExecutor();
        return stringBuilder.toString();
    }

    private String convertJSONToXml(String json) {
        JSONArray jsonArray = JSON.parseArray(json);;
        StringBuilder xmlBuilder = new StringBuilder();

        xmlBuilder.append("<root>");
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject jsonObj = jsonArray.getJSONObject(i);

            String key = jsonObj.getString("key");
            String label = jsonObj.getString("keyName");
            String value = jsonObj.getString("value");

            xmlBuilder.append(String.format("<%s label=\"%s\">%s</%s>",
                    key, label, value, key));
        }
        xmlBuilder.append("</root>");

        return xmlBuilder.toString();
    }

    private static Map<String, Object> convertXmlToMap(String xml) throws Exception {

        XmlMapper xmlMapper = new XmlMapper();
        ObjectMapper objectMapper = new ObjectMapper();

        // 先将 XML 读取为 Map
        Map<?, ?> xmlMap = xmlMapper.readValue(xml, Map.class);

        // 转换为更标准的 Map<String, Object>
        Map<String,Object> map = objectMapper.convertValue(xmlMap, Map.class);
        //特殊处理map格式
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            Map<String,Object> value = (Map<String, Object>) entry.getValue();
            //取出正确的value并设置
            String realValue = String.valueOf(value.get("")).replaceAll("^\\n+","");
            //内容段首移除换行，段末追加换行(会议名称结尾不换行
            entry.setValue(realValue.endsWith("\n") || "meeting_name".equals(entry.getKey()) || "meeting_purpose".equals(entry.getKey()) ? realValue : realValue + "\n");
        }
        return map;
    }

    /**
     * markdown转xml
     * @param markdown
     * @return
     */
    private String extractXmlFromMarkdown(String markdown,String meetingId, String subMeetingId) {
        StringBuffer sb;
        try {
            int start = markdown.indexOf("<");
            if(start == -1){
                processLogService.log(meetingId,subMeetingId,"markdown转xml失败，未输出正确的xml格式，markdown内容："+markdown);
            }
            int end = markdown.lastIndexOf(">") + 1;
            sb = new StringBuffer();
            sb.append("<root>");
            String xml = markdown.substring(start, end).trim().replaceAll("\n\n","\n");
            sb.append(xml);
            sb.append("</root>");
        } catch (Exception e) {
            log.info("markdown转xml，markdown->{}",markdown);
            throw new RuntimeException(e.getMessage());
        }
        return sb.toString();
    }

    private byte[] downloadFile(String url) {
        // 实现文件下载逻辑
        OkHttpClient client = new OkHttpClient();
        Request request = new Request.Builder().url(url).build();
        try {
            Response response = client.newCall(request).execute();
            if (!response.isSuccessful()) throw new IOException("Unexpected code " + response);
            return response.body().bytes();
        }catch (Exception e){
            throw new RuntimeException("下载文件失败", e);
        }
    }


    /**
     * 统计邮件推送情况
     */
    @Override
    public void statisticsEmail(Integer type, Date startTime, Date endTime, HttpServletResponse response) {
        DateTime now = DateUtil.date();
        if (ObjectUtils.isEmpty(startTime) || ObjectUtils.isEmpty(endTime)) {
            if (1== type) {
                startTime = DateUtil.beginOfDay(DateUtil.offset(now, DateField.DAY_OF_MONTH, -2));
                endTime = DateUtil.offset(now, DateField.HOUR_OF_DAY, -2);
            } else {
                startTime = DateUtil.beginOfDay(now);
                endTime = now;
            }
        }

        List<MeetingInfo> meetingInfoList = meetingInfoMapper.selectList(new LambdaQueryWrapper<MeetingInfo>()
//                                                .eq(MeetingInfo::getEmailPushAccess, true)
                .between(MeetingInfo::getStartTime, startTime, endTime));
        Map<String, List<MeetingInfo>> hostUidMeetingInfoMap = meetingInfoList.stream().collect(Collectors.groupingBy(MeetingInfo::getHostUid));
        Map<String, String> hostMap = new HashMap<>();
        meetingInfoList.forEach(meetingInfo -> {
            if (!hostMap.containsKey(meetingInfo.getHostUid())) {
                hostMap.put(meetingInfo.getHostUid(), meetingInfo.getHost());
            }
        });

        // 总计
        Map<String, Object> totalMap = new LinkedHashMap<>();
        totalMap.put("host", "总计");

        List<Map<String, Object>> mapList = new ArrayList<>();
        for (Map.Entry<String, List<MeetingInfo>> entry : hostUidMeetingInfoMap.entrySet()) {
            // 使用linkedHashMap，保证输出到excel中的顺序
            Map<String, Object> map = new LinkedHashMap<>();

            String host = hostMap.get(entry.getKey());
            List<MeetingInfo> meetingList = entry.getValue();
            Integer totalNum = meetingList.size();
            //    没有转录文件会议，需要生成会议纪要的会议，   待处理，    处理中，             纪要生成成功，         推送成功，         推送失败
            Integer  emptyNum=0, emailPushAccessNum=0, newNum=0,  generatErrorNum=0, noteGeneratedNum=0, pushSuccessNum=0, pushErrorNum=0;

            for (MeetingInfo meetingInfo : meetingList) {
                if (meetingInfo.getStatus() == MeetingState.NEW.getCode()) {
                    newNum++;
                } else if (meetingInfo.getStatus() == MeetingState.GENERATE_ERROR.getCode()) {
                    generatErrorNum++;
                } else if (meetingInfo.getStatus() == MeetingState.PUSH_SUCCESS.getCode()) {
                    pushSuccessNum++;
                } else if (meetingInfo.getStatus() == MeetingState.PUSH_ERROR.getCode()) {
                    pushErrorNum++;
                } else if (meetingInfo.getStatus() == MeetingState.EMPTY.getCode()) {
                    emptyNum++;
                }
                if (meetingInfo.getIsGenerated() || meetingInfo.getStatus() == MeetingState.NOTE_GENERATED.getCode()) {
                    noteGeneratedNum++;
                }
                // 有转录文件 且 email_push_access为true，
                if (meetingInfo.getStatus() != MeetingState.EMPTY.getCode() && meetingInfo.getEmailPushAccess()) {
                    emailPushAccessNum++;
                }
            }
            Integer avaliableNum = totalNum - emptyNum;
            map.put("host", host);
            map.put("totalNum", totalNum);
            map.put("avaliableNum", avaliableNum);
            map.put("emptyNum", emptyNum);
            map.put("emailPushAccessNum", emailPushAccessNum);
            map.put("newNum", newNum);
            map.put("generatErrorNum", generatErrorNum);
            map.put("noteGeneratedNum", noteGeneratedNum);
            map.put("pushErrorNum", pushErrorNum);
            map.put("pushSuccessNum", pushSuccessNum);
            mapList.add(map);

            // 计算总计
            totalMap.put("totalNum", Integer.valueOf(totalMap.getOrDefault("totalNum", 0).toString()) + totalNum);
            totalMap.put("avaliableNum", Integer.valueOf(totalMap.getOrDefault("avaliableNum", 0).toString()) + avaliableNum);
            totalMap.put("emptyNum", Integer.valueOf(totalMap.getOrDefault("emptyNum", 0).toString()) + emptyNum);
            totalMap.put("emailPushAccessNum", Integer.valueOf(totalMap.getOrDefault("emailPushAccessNum", 0).toString()) + emailPushAccessNum);
            totalMap.put("newNum", Integer.valueOf(totalMap.getOrDefault("newNum", 0).toString()) + newNum);
            totalMap.put("generatErrorNum", Integer.valueOf(totalMap.getOrDefault("generatErrorNum", 0).toString()) + generatErrorNum);
            totalMap.put("noteGeneratedNum", Integer.valueOf(totalMap.getOrDefault("noteGeneratedNum", 0).toString()) + noteGeneratedNum);
            totalMap.put("pushErrorNum", Integer.valueOf(totalMap.getOrDefault("pushErrorNum", 0).toString()) + pushErrorNum);
            totalMap.put("pushSuccessNum", Integer.valueOf(totalMap.getOrDefault("pushSuccessNum", 0).toString()) + pushSuccessNum);
        }
        mapList.add(totalMap);

        // 导出
        OutputStream outputStream = null;
        ExcelWriter writer = ExcelUtil.getWriter(true);
        try {
            String title = String.format("会议纪要推送统计表_%s-%s", DateUtil.format(startTime, "yyyyMMdd"), DateUtil.format(endTime, "yyyyMMdd"));

            outputStream = response.getOutputStream();
            String fileName = String.format("%s.xlsx", title);
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet;charset=utf-8");
            response.setHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode(fileName, "utf8"));


            writer.addHeaderAlias("host", "主持人");
            writer.addHeaderAlias("totalNum", "总会议");
            writer.addHeaderAlias("avaliableNum", "有效会议");
            writer.addHeaderAlias("emptyNum", "没有转录文件");
            writer.addHeaderAlias("emailPushAccessNum", "需生成纪要的会议总数");
            writer.addHeaderAlias("newNum", "待处理");
            writer.addHeaderAlias("generatErrorNum", "处理中");
            writer.addHeaderAlias("noteGeneratedNum", "纪要生成成功");
            writer.addHeaderAlias("pushSuccessNum", "纪要推送成功");
            writer.addHeaderAlias("pushErrorNum", "纪要推送失败");
            // 合并单元格后的标题行，使用默认标题样式
            writer.merge(writer.getHeaderAlias().size() - 1, title);
            // 一次性写出内容，使用默认样式，强制输出标题
            writer.write(mapList, true);
            writer.flush(outputStream);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            writer.close();
            IoUtil.close(outputStream);
        }
    }

    private static Map<String, String> DEPT_MAP = new HashMap<>(20);
    static {
        DEPT_MAP.put("10340", "集团总部");
        DEPT_MAP.put("602700", "能化板块");
        DEPT_MAP.put("616859", "空港板块");
        DEPT_MAP.put("616766", "集装箱集团");
        DEPT_MAP.put("616844", "车辆板块");
        DEPT_MAP.put("616860", "海工板块");
        DEPT_MAP.put("616861", "产城板块");
        DEPT_MAP.put("640071", "物流板块");
        DEPT_MAP.put("617560", "融资租赁贵公司");
        DEPT_MAP.put("640075", "财务公司");
        DEPT_MAP.put("641858", "其他类型企业");
        DEPT_MAP.put("707435", "集团创新企业");
        DEPT_MAP.put("686701", "中集海工");
        DEPT_MAP.put("710972", "中集资本");
        DEPT_MAP.put("711091", "中集载具");
        DEPT_MAP.put("698023", "中集同创");

    }


    /**
     * 统计邮件推送情况
     */
    @Override
    public boolean statisticsEmail(String title, Date startTime, Date endTime, OutputStream outputStream) {
        List<MeetingInfo> meetingInfoList = meetingInfoMapper.selectList(new LambdaQueryWrapper<MeetingInfo>()
                                                .between(MeetingInfo::getStartTime, startTime, endTime));
        log.info("会议总量: {}", meetingInfoList.size());
        if (meetingInfoList.size() == 0) {
            log.info("时间段内没有会议");
            return false;
        }

        // 查询工号以及部门
        List<String> uids = meetingInfoList.stream().map(MeetingInfo::getHostUid).distinct().collect(Collectors.toList());
        List<UserId> userIds = userIdMapper.selectList(new LambdaQueryWrapper<UserId>().in(UserId::getTid, uids));

        Map<String, String> idMap = userIds.stream().collect(Collectors.toMap(UserId::getTid, UserId::getWid));
        Map<String, List<String>> deptMap = new HashMap<>();
        for (UserId userId : userIds) {
            List<String> deptPathByUserIdAndInSet = sysUserSyncService.getDeptPathByUserIdAndInSet(userId.getWid());
            deptMap.put(userId.getWid(), deptPathByUserIdAndInSet);
        }

        Map<String, List<MeetingInfo>> hostUidMeetingInfoMap = new HashMap<>();
        for (MeetingInfo meetingInfo : meetingInfoList) {
            for (String path : deptMap.get(idMap.get(meetingInfo.getHostUid()))) {
                for (String deptId : DEPT_MAP.keySet()) {
                    if (path.contains(deptId)) {
                        List<MeetingInfo> meetingInfos = hostUidMeetingInfoMap.get(DEPT_MAP.get(deptId));
                        if (CollUtil.isEmpty(meetingInfos)) {
                            meetingInfos = new ArrayList<>();
                        }
                        meetingInfos.add(meetingInfo);
                        hostUidMeetingInfoMap.put(DEPT_MAP.get(deptId), meetingInfos);
                        break;
                    }
                }
            }
        }

        Map<String, String> hostMap = new HashMap<>();
        meetingInfoList.forEach(meetingInfo -> {
            if (!hostMap.containsKey(meetingInfo.getHostUid())) {
                hostMap.put(meetingInfo.getHostUid(), meetingInfo.getHost());
            }
        });

        // 总计
        Map<String, Object> totalMap = new LinkedHashMap<>();
        totalMap.put("host", "总计");

        List<Map<String, Object>> mapList = new ArrayList<>();

        for (Map.Entry<String, List<MeetingInfo>> entry : hostUidMeetingInfoMap.entrySet()) {
            // 使用linkedHashMap，保证输出到excel中的顺序
            Map<String, Object> map = new LinkedHashMap<>();

            List<MeetingInfo> meetingList = entry.getValue();
            Integer totalNum = meetingList.size();
            //    没有转录文件会议，需要生成会议纪要的会议，   待处理，    处理中，             纪要生成成功，         推送成功，         推送失败
            Integer  emptyNum=0, emailPushAccessNum=0, newNum=0,  generatErrorNum=0, noteGeneratedNum=0, pushSuccessNum=0, pushErrorNum=0;

            for (MeetingInfo meetingInfo : meetingList) {
                if (meetingInfo.getStatus() == MeetingState.GENERATE_ERROR.getCode()) {
                    generatErrorNum++;
                } else if (meetingInfo.getStatus() == MeetingState.PUSH_SUCCESS.getCode()) {
                    pushSuccessNum++;
                } else if (meetingInfo.getStatus() == MeetingState.PUSH_ERROR.getCode()) {
                    pushErrorNum++;
                } else if (meetingInfo.getStatus() == MeetingState.EMPTY.getCode()) {
                    emptyNum++;
                } else if (meetingInfo.getStatus() == MeetingState.NEW.getCode() && meetingInfo.getEmailPushAccess()) {
                    newNum++;
                }
                if (meetingInfo.getIsGenerated() || meetingInfo.getStatus() == MeetingState.NOTE_GENERATED.getCode()) {
                    noteGeneratedNum++;
                }
                // 有转录文件 且 email_push_access为true，
                if (meetingInfo.getStatus() != MeetingState.EMPTY.getCode() && meetingInfo.getEmailPushAccess()) {
                    emailPushAccessNum++;
                }
            }
            Integer avaliableNum = totalNum - emptyNum;
            map.put("host", entry.getKey());
            map.put("totalNum", totalNum);
            map.put("avaliableNum", avaliableNum);
            map.put("emptyNum", emptyNum);
            map.put("emailPushAccessNum", emailPushAccessNum);
            map.put("newNum", newNum);
            map.put("generatErrorNum", generatErrorNum);
            map.put("noteGeneratedNum", noteGeneratedNum);
            map.put("pushErrorNum", pushErrorNum);
            map.put("pushSuccessNum", pushSuccessNum);
            mapList.add(map);

            // 计算总计
            totalMap.put("totalNum", Integer.valueOf(totalMap.getOrDefault("totalNum", 0).toString()) + totalNum);
            totalMap.put("avaliableNum", Integer.valueOf(totalMap.getOrDefault("avaliableNum", 0).toString()) + avaliableNum);
            totalMap.put("emptyNum", Integer.valueOf(totalMap.getOrDefault("emptyNum", 0).toString()) + emptyNum);
            totalMap.put("emailPushAccessNum", Integer.valueOf(totalMap.getOrDefault("emailPushAccessNum", 0).toString()) + emailPushAccessNum);
            totalMap.put("newNum", Integer.valueOf(totalMap.getOrDefault("newNum", 0).toString()) + newNum);
            totalMap.put("generatErrorNum", Integer.valueOf(totalMap.getOrDefault("generatErrorNum", 0).toString()) + generatErrorNum);
            totalMap.put("noteGeneratedNum", Integer.valueOf(totalMap.getOrDefault("noteGeneratedNum", 0).toString()) + noteGeneratedNum);
            totalMap.put("pushErrorNum", Integer.valueOf(totalMap.getOrDefault("pushErrorNum", 0).toString()) + pushErrorNum);
            totalMap.put("pushSuccessNum", Integer.valueOf(totalMap.getOrDefault("pushSuccessNum", 0).toString()) + pushSuccessNum);
        }
        mapList.add(totalMap);

        // 导出
        ExcelWriter writer = ExcelUtil.getWriter(true);
        try {
            writer.addHeaderAlias("host", "部门");
            writer.addHeaderAlias("totalNum", "总会议");
            writer.addHeaderAlias("avaliableNum", "有效会议");
            writer.addHeaderAlias("emptyNum", "没有转录文件");
            writer.addHeaderAlias("emailPushAccessNum", "需生成纪要的会议总数");
            writer.addHeaderAlias("newNum", "待处理");
            writer.addHeaderAlias("generatErrorNum", "处理中");
            writer.addHeaderAlias("noteGeneratedNum", "纪要生成成功");
            writer.addHeaderAlias("pushSuccessNum", "纪要推送成功");
            writer.addHeaderAlias("pushErrorNum", "纪要推送失败");
            // 合并单元格后的标题行，使用默认标题样式
            writer.merge(writer.getHeaderAlias().size() - 1, title);
            // 一次性写出内容，使用默认样式，强制输出标题
            writer.write(mapList, true);
            writer.flush(outputStream);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            writer.close();
            IoUtil.close(outputStream);
        }
        return true;
    }
}
